/*

	Align to Point v1.9
	Matt Gorner
	03 December 2011
	www.pixsim.co.uk

	Revision History:

	v1.9 - Complete overhaul of UI and functionality

	v1.8 - Added ability to align selected geometry to world origin using the 'Stored Bounding Box' centre (set by the 'Store_XYZ_Coords.ls' script) as a datum

	v1.7 - First Release



	Description:

	Aligns selected geometry (using point mode) to a specified point elsewhere on the geometry, world origin (0,0,0) or a stored location using the Coord Tools pack



	Usage:

	1) On the geometry you want to move, select 1 point that will be the move 'anchor' or datum point

	2) Now select the rest of the points on the remaining geometry you want to move. (Use 'Select Connected ']' key to quickly selected points on the same object or lasso select for other unconnected geometry)

	3) Lastly, select the point which you want the first point (and subsequent points) you selected to move to (only needed if you intend to use the Align to Last Selected Point option)

	4) Run the 'Align to Point' LScript:

		- Moxe X, Move Y and Move Z = Which axis will be moved

		- Source: First Point or Bounding Box = Use the first selected point as the datum, or the selection bounding box

		- Center All: (Only needed when in Bounding Box mode) will center the selection to the chosen 'Align to' mode

		- BBox X / Y Z = (Only available when in 'Center All' is unchecked) allows you to use the extents (outer limits) of the selection bounding box to align to

		- Align to: = The alignment mode you wish to use:

			'Last Selected Point'				= Is exactly that, the last selection point you chose before running the script
			'World Origin (0,0,0)				= The X:0 Y:0 Z:0 center of the modelling environment
			'Coord Tool Pack Stored Location'	= Uses the stored location in memory when you ran the 'Store_XYZ_Coords.ls' script (part of the Coord Tool pack available from www.pixsim.co.uk

*/

@script modeler
@warnings
@version 2.3
@name "Align to Point"

// Globals
gad_source_mode, gad_center, gad_align_mode, gad_bb_align_x, gad_bb_align_y, gad_bb_align_z, gad_deselect;

main
{
	// Grab the points the user has _actually_ selected, and not using the rule 'if nothing is selected everything is selected'
	selmode( DIRECT );

	// Get total number of points selected
	selected_points = editbegin();

	// Nothing selected?
	if (selected_points == 0)
	{
		// User might be in poly or edge mode, let's not deny the use of these!
		error("Align to Point: There are no points selected.");
	}

	// Store the last point selected so we can deselect it later
	last_point 		= points[selected_points];

	// Pull previously used values for requester
	move_x			= recall("ATP2_move_x",         1);
	move_y 			= recall("ATP2_move_y",         1);
	move_z 			= recall("ATP2_move_z",         1);
    source_mode     = recall("ATP2_source_mode",    1);
    centered 		= recall("ATP2_centered",       1);
	alignment_x 	= recall("ATP2_alignment_x",    2);
	alignment_y     = recall("ATP2_alignment_y",    3);
    alignment_z		= recall("ATP2_alignment_z",    2);
    align_mode 		= recall("ATP2_align_mode",     1);
    deselect 		= recall("ATP2_deselect",       0);

	// Recall globally stored coordinates from 'Store_XYZ_Coords.ls' script (default to 0 if not set)
	CTP_stored_x 		= number(globalrecall("stored_x", 0));
	CTP_stored_y 		= number(globalrecall("stored_y", 0));
	CTP_stored_z 		= number(globalrecall("stored_z", 0));

	// Extract the FIRST points XYZ coordinates
	first_point_x 	= pointinfo(points[1]).x;
	first_point_y 	= pointinfo(points[1]).y;
	first_point_z 	= pointinfo(points[1]).z;

	// Extract the LAST points XYZ coordinates
	last_point_x 	= pointinfo(points[selected_points]).x;
	last_point_y 	= pointinfo(points[selected_points]).y;
	last_point_z 	= pointinfo(points[selected_points]).z;

	// Okay we have everything we need so can end 'edit mode'
	editend();

	// User interface layout variables

    gad_text_offset = 60;
    gad_x_offset    = 0;
    gad_y_offset    = 2;
    gad_w           = gad_text_offset + 75;
    gad_h           = 19;
    num_gads        = 5;
    num_spacers     = 2;
    ui_spacing      = 5;
    spacer_offset   = gad_h / 2;
    ui_offset_x     = 0;
    ui_offset_y     = ui_spacing * 2;
    ui_row          = 0;
    ui_row_offset   = gad_h + ui_spacing;
    ui_window_w     = 320;
    ui_window_h     = 320;

	// Setup interface . . .
 	reqbegin(" Align to Point v1.9 ..." );
	reqsize(ui_window_w, ui_window_h);

	// Setup the gadgets ...
	gad_movex = ctlcheckbox("Move X", move_x);
	ctlposition(gad_movex, gad_x_offset, gad_y_offset + ui_offset_y, gad_w, gad_h, gad_text_offset);

 	gad_movey 	= ctlcheckbox("Move Y", move_y);
 	ctlposition(gad_movey, gad_x_offset + (gad_w - gad_text_offset) + ui_spacing, gad_y_offset + ui_offset_y, gad_w, gad_h, gad_text_offset);
    
 	gad_movez 	= ctlcheckbox("Move Z", move_z);
 	ctlposition(gad_movez, gad_x_offset + (gad_w - gad_text_offset) * 2 + (ui_spacing * 2), gad_y_offset + ui_offset_y, gad_w, gad_h, gad_text_offset);

  	ui_offset_y += ui_row_offset + spacer_offset;
    
	gad_sep_01 = ctlsep();
	ctlposition(gad_sep_01, -2, gad_y_offset + ui_offset_y, ui_window_w + 4);

    ui_offset_y += ui_row_offset - spacer_offset;
    
	gad_source_mode = ctlchoice("Source", source_mode, @ "First Point", "Bounding Box"@, false);
	ctlposition(gad_source_mode, gad_x_offset, gad_y_offset + ui_offset_y, gad_w * 2.2 - 2, gad_h, gad_text_offset);     
    
    ui_offset_y += ui_row_offset;

	gad_center = ctlcheckbox("Center All", centered);
	ctlposition(gad_center, gad_x_offset, gad_y_offset + ui_offset_y, gad_w * 2.2 - 2, gad_h, gad_text_offset);
	gad_center.active(source_mode == 1? false : true);

    ui_offset_y += ui_row_offset;

	gad_bb_align_x = ctlchoice("BBox X", alignment_x, @ "( - ) Left", "Center", "( + ) Right" @, false);
	ctlposition(gad_bb_align_x, gad_x_offset, gad_y_offset + ui_offset_y, gad_w * 2.2 - 2, gad_h, gad_text_offset);

    ui_offset_y += ui_row_offset - ui_spacing - 1;

	gad_bb_align_y = ctlchoice("Y", alignment_y, @ "( + ) Top", "Center", "( - ) Bottom" @, false);
	ctlposition(gad_bb_align_y, gad_x_offset, gad_y_offset + ui_offset_y, gad_w * 2.2 - 2, gad_h, gad_text_offset);

    ui_offset_y += ui_row_offset - ui_spacing - 1;

	gad_bb_align_z = ctlchoice("Z", alignment_z, @ "( + ) Front", "Center", "( - ) Back" @, false);
	ctlposition(gad_bb_align_z, gad_x_offset, gad_y_offset + ui_offset_y, gad_w * 2.2 - 2, gad_h, gad_text_offset);

    ui_offset_y += ui_row_offset + ui_spacing + 2;

	gad_sep_02 = ctlsep();
	ctlposition(gad_sep_02, -2, gad_y_offset + ui_offset_y, ui_window_w + 4);

    ui_offset_y += ui_row_offset - spacer_offset;

	gad_align_mode 		= ctlchoice("Align to", align_mode, @ "Last Selected Point", "World Origin (0,0,0)","Coord Tool Pack Stored Location" @, true);
	ctlposition(gad_align_mode, gad_x_offset, gad_y_offset + ui_offset_y, gad_w * 2.2 - 2, gad_h * 3, gad_text_offset);

    ui_offset_y += ui_row_offset;
    ui_offset_y += ui_row_offset;
    ui_offset_y += ui_row_offset;

	gad_deselect = ctlcheckbox("Deselect Last Point", deselect);
	ctlposition(gad_deselect, gad_x_offset, gad_y_offset + ui_offset_y, gad_w * 2.2 - 2, gad_h, gad_text_offset);

    // For gadget ghosting logic
    ctlrefresh( gad_source_mode,    "update_gadgets");
    ctlrefresh( gad_center,         "update_gadgets");
    ctlrefresh( gad_align_mode,     "update_gadgets");

	// Set initial ghosting
	if(source_mode == 1 || centered == 1)
	{
	    gad_bb_align_x.active(false);
	    gad_bb_align_y.active(false);
	    gad_bb_align_z.active(false);
	}else
	{
	    gad_bb_align_x.active(true);
	    gad_bb_align_y.active(true);
	    gad_bb_align_z.active(true);
    }

	gad_deselect.active( (align_mode == 1)? true : false);


	// Show interface and gadgets (quit if it fails)
	return if !reqpost();

		// Grab the user input from the gadgets
		move_x 		= getvalue( gad_movex       );
		move_y 		= getvalue( gad_movey       );
		move_z 		= getvalue( gad_movez       );
        source_mode = getvalue( gad_source_mode );
        centered 	= getvalue( gad_center      );
		align_mode 	= getvalue( gad_align_mode  );
		alignment_x	= getvalue( gad_bb_align_x  );
        alignment_y	= getvalue( gad_bb_align_y  );
		alignment_z	= getvalue( gad_bb_align_z  );
        deselect	= getvalue( gad_deselect    );

	// Close the requester
	reqend();

	// Store our variables for the next time the script is run (within one LW session, but not if you quit LW)
	store("ATP2_move_x",        move_x      );
	store("ATP2_move_y",        move_y      );
	store("ATP2_move_z",        move_z      );
	store("ATP2_align_mode",    align_mode  );
	store("ATP2_source_mode",   source_mode );
    store("ATP2_centered",      centered    );
	store("ATP2_alignment_x",   alignment_x );
	store("ATP2_alignment_y",   alignment_y );
	store("ATP2_alignment_z",   alignment_z );
	store("ATP2_deselect",      deselect    );

	// Do stuff . . .
	
	// Except if no axis are selected at all!
	if (!move_x && !move_y && !move_z)
		return;
    
    undogroupbegin();

	// Deselect last point only if we are using last selected point alignment
 	if( align_mode == 1 )
	{
        // Unless the user has specifically said don't do it!
        if( deselect )
            selpoint(CLEAR, POINTID, last_point );
 	}

	// Change to USER selection mode because the boundingbox() function seems to need it
	selmode( USER );

	// Find centre and bounding box extents of remaining selection
	editbegin();
		selection_centre    = center(boundingbox());
        (bb_lo, bb_hi)      = boundingbox();
	editend();
       
    // Work out source point location
    if (source_mode == 1) // Use First Point of Selection
    {
		source_point = <first_point_x,first_point_y,first_point_z>;

    }else // Use Bounding Box of Selection
    {
    
        if(centered) // Use Bounding Box Center of Selection
        {
            source_point = <selection_centre.x,selection_centre.y,selection_centre.z>;
            
        }else // User Selected Bounding Box Extents of Selection
        {
              
            source_point = <    // - Left  |   Center  |   + Right
                                ((alignment_x == 1)? bb_lo.x : (alignment_x == 2)? selection_centre.x : bb_hi.x),
                                // + Top    |   Center  |   - Bottom
                                ((alignment_y == 1)? bb_hi.y : (alignment_y == 2)? selection_centre.y : bb_lo.y),
                                // + Front  |   Center  |   - Back
                                ((alignment_z == 1)? bb_hi.z : (alignment_z == 2)? selection_centre.z : bb_lo.z)>;

        }
        
    }
    
    
	// Work out move distance based on user selected mode
	switch( align_mode )
	{
		case 1: // First Selected Point to Last Point
                difference_x = (last_point_x - source_point.x);
                difference_y = (last_point_y - source_point.y);
                difference_z = (last_point_z - source_point.z);
      	break;

		case 2: // Selection to World Origin
                difference_x = (0 - source_point.x);
                difference_y = (0 - source_point.y);
                difference_z = (0 - source_point.z);
      	break;

		case 3: // Selection to Coord Tool Pack Stored Location
                difference_x = (CTP_stored_x - source_point.x);
                difference_y = (CTP_stored_y - source_point.y);
                difference_z = (CTP_stored_z - source_point.z);
      	break;

		default:
      	break;
	}

    
	// Check which axis are required, reset the move difference if not selected
	if (!move_x)
		difference_x = 0;
	if (!move_y)
		difference_y = 0;
	if (!move_z)
        difference_z = 0;
        
	// Move selection
    move(difference_x, difference_y, difference_z);
    
    // Reselect last point because user requested it
 	if ( deselect == false )
	{
	 	selpoint(SET, POINTID, last_point );
 	}
    
    undogroupend();

	// Finished!
	return;
} // End

update_gadgets: value
{
    if(gad_source_mode.value == 1)
    {
		gad_center.active(false);

        gad_bb_align_x.active(false);
        gad_bb_align_y.active(false);
        gad_bb_align_z.active(false);
    }else
    {
    	gad_center.active(true);

        if(gad_center.value == 1)
        {
            gad_bb_align_x.active(false);
            gad_bb_align_y.active(false);
            gad_bb_align_z.active(false);
        }else
        {
            gad_bb_align_x.active(true);
            gad_bb_align_y.active(true);
            gad_bb_align_z.active(true);        
        }            
    }
    
    if(gad_align_mode.value == 1)
    {
        gad_deselect.active(true);
    }else
    {
        gad_deselect.active(false);    
    }
    
    requpdate();
}
